#!/usr/bin/python3

from os import mkdir
from os.path import isdir, isfile, join

from markup import markup_article
from span import *
from util import *

__all__ = 'get_menu html_menu walk_menu'.split()

# Primary use for get_menu():
#
# Open the menu file and slurp into a collection like this:
#
#    [ ('Some topic', ['First article', 'Further article']),
#      ('*Private topic', ['Undisplayed article']),
#      ('Another topic', ['*Undisplayed draft', 'Strange article']) ]
#
# When a topic name starts with '*', it and its articles are unadvertised.
# When an article name starts with '*', that article is unadvertised.
#
# Unadvertised items do not appear in menus on advertised pages. That way,
# the website may have various "drafts" that can be seen by their writers
# (either via direct links or from menus within the unadvertised realm)
# but shouldn't be noticed by regular visitors.
#
# Secondary use for get_menu():
#
# Fill in quickLinks[], which is a dict that looks like:
#
#    { 'Some topic': '<a href="/...">...</a>',
#       'Some article': '<a href="/.../...">...</a>' }
#
def get_menu(fname):
    M = []
    topic_fname: str = None
    lines = slurp(fname)

    for line in lines.splitlines():
        topic = not line[:1].isspace()
        line = line.strip().split(';')[0]
        nostar = line[1:] if line.startswith('*') else line
        if not line:
            continue

        # We have either a topic or an article.
        if topic:
            M.append((line, []))
            topic_fname = cleanup_filename(nostar)
            ql_set(nostar, '<a href="/%s">%s</a>' % (topic_fname, nostar))
        else:
            if not M:
                oops('first line of [%s] must be a topic' % menufile)
            M[-1][1].append(line)
            ql_set(nostar, '<a href="/%s/%s">%s</a>' % (
                    topic_fname, cleanup_filename(nostar), nostar))

    return M

# Convert the menu structure into HTML to display it.
#
# 'M' is the menu to show.
# 'this_topic' and 'this_art' let us highlight one menu entry.
# They also determine whether unadvertised items should be shown.
def html_menu(
    M: list[tuple[str, list[str]]],
    this_topic: str,
    this_art: str) -> str:

    show_all = this_topic.startswith('*') or this_art.startswith('*')

    s = '<ul class="menu1">\n'
    for topic, arts in M:

        if not show_all and topic.startswith('*'):
            continue                    # Hide because unadvertised.

        if topic == this_topic and 'index.html' == this_art:
            mark = ' id="mark"'         # Highlight this menu item.
        else:
            mark = ''

        # Sanitize filenames.
        topic_ = cleanup_filename(topic)

        # Don't display '*' in draft menu items. Instead, change color.
        if topic.startswith('*'):
            top, draft = topic[1:], ' class="draft"'
        else:
            top, draft = topic, ''

        s += '    <li>\n'
        s += '        <a href="/%s/"%s%s>%s</a>\n' % (topic_, mark, draft, top)
        if arts:
            s += '        <ul class="menu2">\n'
            for art in arts:
                if not show_all and art.startswith('*'):
                    continue            # Hide because unadvertised.
                if art == 'index.html':
                    continue            # User clicks topic link instead.
                if topic == this_topic and art == this_art:
                    mark = ' id="mark"' # Highlight this menu item.
                else:
                    mark = ''
                art_ = cleanup_filename(art)

                # Don't display '*' in draft menu items. Instead, change color.
                if art.startswith('*'):
                    art2, draft = art[1:], ' class="draft"'
                else:
                    art2, draft = art, ''
                if topic.startswith('*'):
                    draft = ' class="draft"'

                s += '            <li><a href="/%s/%s"%s%s>%s</a></li>\n' % (topic_, art_, mark, draft, art2)
            s += '        </ul>\n'
        s += '    </li>\n'
    s += '</ul>\n'
    return s

# Make a filename from a topic or article name.
def cleanup_filename(s: str) -> str:
    ok = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_.'

    if s.startswith('*'):
        s = s[1:]           # The unadvertised mark is not part of filename.

    # Make non-URL symbols a _ instead, no dupes, no leading/trailing.
    t = ''.join(c if c in ok else ' ' for c in s)
    t = ' '.join(t.split()).replace(' ', '_')

    # No trailing _s.
    while t.endswith('_'):
        t = t[:-1]
    return t

# Walk through the menu collection and instantiate stuff.
def walk_menu(M, indir, outdir):

    print('    processing articles...')

    # Process the master index.html for the whole site (it's not in the menu).
    one_article('', 'index.html', indir, outdir, M)

    for (topic, articles) in M:

        # Create the output directory.
        t_fname = cleanup_filename(topic)
        mkdir('%s/%s' % (outdir, t_fname))

        # Process index.html for that topic.
        one_article(topic, 'index.html', indir, outdir, M)

        # Process the articles.
        for art in articles:
            one_article(topic, art, indir, outdir, M)

# walk_menu helper -- main purpose is to catch the master index.html
def one_article(
    topic: str,
    article: str,
    indir: str,
    outdir: str,
    M: list[tuple[str, list[str]]]
):

    t_fname = cleanup_filename(topic)
    a_fname = cleanup_filename(article)

    # path for topic/article
    top_art = '%s/%s' % (t_fname, a_fname) if t_fname else a_fname

    # path for in and out
    inpath = '%s/%s' % (indir, top_art)
    outpath = '%s/%s' % (outdir, top_art)

    # Read the file.
    try:
        text = slurp(inpath)
    except:
        oops('cannot read file [%s]' % inpath)

    # Process it as an article.
    text = markup_article(text)

    # Draw up the menu sidebar. Unpublished material will include
    # unpublished menu items in addition to regular menu items.
    menu_html = html_menu(M, topic, article)
    text = text.replace('/MENU\\', menu_html)

    # Write the output.
    try:
        open(outpath, 'w').write(text)
    except:
        oops('unable to write file [%s]', fname)

