#!/usr/bin/python3

__all__ = 'ql_get ql_in ql_set span xmlquote'.split()

from util import oops, wherein

# quick_links lets the user markup something that's in the menu
# simply by its menu text; e.g., [Fibonacci numbers] instead of
# [Fibonacci_numbers|Fibonacci numbers]. That it's global here is
# a kludge; otherwise it would have to pass through many function
# calls unchanged.
quick_links: dict[str, str] = {}

# Too lazy to subclass today, thank you.
def ql_get(key):
    return quick_links[key.casefold()]

def ql_set(key, val):
    quick_links[key.casefold()] = val

def ql_in(key):
    return key.casefold() in quick_links

def ql_iter():
    return quick_links.__iter__()

# Handle spans. It needs its own module to avoid a circular import.
#
# "Current special" dict 'cs' maps actions onto special characters
# that achieve them. The key is the default character for each action,
# and the value is the character that should be searched and replaced.
#
# TODO This would detect nesting errors like `*example`*
#      if it were rewritten to be recursive. Right now,
#      invalid HTML is generated for invalid input.
def span(cs: dict[str, str], s: str) -> str:
    # Because multiple spaces in HTML are extraneous, they go away here.
    s = s.lstrip()
    while True:
        was, s = len(s), s.replace('  ', ' ')
        if was == len(s):
            break                       # no replacment happened

    s = xmlquote(s)

    # ~ -> minus sign
    x = cs['~']
    s = s.replace(x, '&minus;')

    # --- -> em dash                    # order matters
    x = 3 * cs['-']
    s = s.replace(x, '&mdash;')

    # -- -> en dash                     # order matters
    x = 2 * cs['-']
    s = s.replace(x, '&ndash;')

    # "text" --> &ldquo;text&rdquo;
    x = cs['"']
    s = replace2(s, x, x, '&ldquo;', '&rdquo;')

    # Biden's --> Biden&rsquo;s
    x = cs["'"]
    s = s.replace(x, '&rsquo;')

    # ****text**** --> <sup>text</sup> # order matters
    x = 4 * cs['*']
    s = replace2(s, x, x, '<sup>', '</sup>')

    # ***text*** --> <b><i>text</i></b> # order matters
    x = 3 * cs['*']
    s = replace2(s, x, x, '<b><i>', '</i></b>')

    # **text** --> <b>text</b>          # order matters
    x = 2 * cs['*']
    s = replace2(s, x, x, '<b>', '</b>')

    # *text* --> <i>text</i>            # order matters
    x = cs['*']
    s = replace2(s, x, x, '<i>', '</i>')

    # `text` --> <code>text</code>
    x = cs['`']
    s = replace2(s, x, x, '<code>', '</code>')

    # [url]
    # We use a loop to find and replace all of the URLs.
    x, y = cs['['], cs[']']
    while True:
        abc = split2(s, x, y)
        if abc is None:
            break                       # no URL remains to process
        a, b, c = abc

        # a and c are prefix and suffix. b has four cases:
        #
        # 1. menu item: The menu system already generated the <a> for us.
        # 2. URL
        # 3. URL|text to show
        # 4. URL|text to show|mouseover text

        if '/' not in b:
            if ql_in('*' + b):
                s = '%s%s%s' % (a, ql_get('*' + b), c)
                continue
            elif ql_in(b):
                s = '%s%s%s' % (a, ql_get(b), c)
                continue
            else:
                for l in ql_iter():
                    print(l)
                oops('probable bad link: "%s"' % b)

        # External links should prefer HTTPS and open a new tab.
        if 'http:' in b:
            print('warn in %s: http: protocol' % wherein())
        target = ' target="_blank"' if 'https:' in b else ''

        if '|' not in b:
            # just URL
            s = '%s<a href="%s"%s>%s</a>%s' % (a, b, target, b, c)
            continue

        b1, b2 = b.split('|', 1)
        if '|' not in b2:
            # URL and link text
            s = '%s<a href="%s"%s>%s</a>%s' % (a, b1, target, b2, c)
            continue
        b2, b3 = b2.split('|', 1)
        if '|' in b3:
            print('warn: | in mouseover text')
        if not b2:
            b2 = b1                     # mouseover w/o link text change

        # URL, link text, and mouseover text
        s = '%s<a href="%s" title="%s"%s>%s</a>%s' % (a, b1, b3, target, b2, c)

    return s

# Split a string at two delimiters. They can have any nonzero length.
# The left delimiter must occur first, then the right delimiter
# without another left delimiter first. Returns None or a 3-tuple.
# Issues warnings.
def split2(s: str, lfind: str, rfind: str) -> None | tuple[str, str, str]:
    a_bc = s.split(lfind, 1)

    if len(a_bc) == 1:              # no match on first delim, so no change
        if rfind in s:
            print('warn in %s: unmatched %s' % (wherein(), rfind))
        return None

    a, bc = a_bc
    b_c = bc.split(rfind, 1)

    if len(b_c) == 1:
        print('warn in %s: unmatched %s' % (wherein(), lfind))
        return None
    b, c = b_c
    if lfind in b:
        print('warn in %s: unmatched %s' % (wherein(), lfind))
        return None

    return a, b, c

# Replace two delimiters in a string.
#
#     replace2('this (:is:) cool', '(:', ':)', '<', '>') --> 'this <is> cool'
#
def replace2(s: str, lfind: str, rfind: str, lrepl: str, rrepl: str) -> str:
    while True:
        abc = split2(s, lfind, rfind)
        if abc is None:
            break
        a, b, c = abc
        s = a + lrepl + b + rrepl + c

    return s

# quote the XML mandatories
def xmlquote(s: str) -> str:
    s = s.replace('&', '&amp;')     # order matters!
    s = s.replace('<', '&lt;')
    s = s.replace('>', '&gt;')
    return s

